using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;
using Microsoft.Xna.Framework.Net;
using Microsoft.Xna.Framework.Storage;
using CashierCrisis.Managers;
using CashierCrisis.Scenes;
using CashierCrisis.UserInterface;
using CashierCrisis.Core;
using System.Xml.Serialization;
using System.IO;
using CashierCrisis.Graphs;

namespace CashierCrisis
{
    /// <summary>
    /// This is the main type for your game
    /// </summary>
    public class CashierGame : Microsoft.Xna.Framework.Game
    {
        GraphicsDeviceManager graphics;
        SceneManager sceneManager;
        UIManager uiManager;
        EntityManager entityManager;
        InputManager inputManager;
        FontManager fontManager;
        SpriteBatch spriteBatch;
        SpriteFont defaultFont;

        SparseGraph<PathNode, GraphEdge> graph;


      

        public CashierGame()
        {
            graphics = new GraphicsDeviceManager(this);
            Content.RootDirectory = "Content";
            Components.Add(new GamerServicesComponent(this));
        }

        private void GraphUnitTest()
        {
             Log.WriteLine(LogCategory.General, "---Graph Test Start---");

             PathNode node1 = new PathNode(Vector2.Zero);
             node1.Index = 0;

             PathNode node2 = new PathNode(new Vector2(100,0));
             node2.Index = 1;

             PathNode node3 = new PathNode(new Vector2(200,0));
             node3.Index = 2;

             GraphEdge edge1 = new GraphEdge(0, 1, 1.0);
             GraphEdge edge2 = new GraphEdge(1, 2, 1.0);

             graph.AddNode(node1);
             graph.AddNode(node2);
             graph.AddNode(node3);
             graph.AddEdge(edge1);
             graph.AddEdge(edge2);
             
             Log.WriteLine(LogCategory.General, string.Format("Graph Node Count {0}", graph.NodeCount.ToString()));
             Log.WriteLine(LogCategory.General,string.Format("Contains Index {0}:{1}",
                 node1.Index.ToString(), graph.ContainsNode(node1.Index).ToString()));
             Log.WriteLine(LogCategory.General,string.Format("Get Edge Cost :{0}", graph.GetEdge(0, 1).Cost.ToString()));

             AStarGraphSearch<PathNode, GraphEdge> astar = 
                 new AStarGraphSearch<PathNode, GraphEdge>(graph, 0, 2, new HeuristicEuclid()); 
            
            List<int> path = astar.PathToTarget();
            
            for (int i = 0; i < path.Count; i++)
            {
                Log.WriteLine(LogCategory.General,string.Format("Path Node Index: {0} ", path[i].ToString()));
            }

            foreach (GraphEdge edge in graph.Edges())
            {
                Log.WriteLine(LogCategory.General, string.Format(" Edge From:{0} To:{1}", edge.From, edge.To));
            }

            PathPlanner.Graph = graph;
       
            

            List<Vector2> vPath = PathPlanner.CreatePathToPosition(Vector2.Zero, new Vector2(200,0));
            for (int i = 0; i < vPath.Count; i++)
            {
                Log.WriteLine(LogCategory.General, vPath[i].ToString());
            }
            

            Log.WriteLine(LogCategory.General, string.Format("Total Cost {0}", astar.CostToTarget.ToString()));
            Log.WriteLine(LogCategory.General, "---Graph Test End---");
        }

        private void GraphUnitTestTwo()
        {
            Log.WriteLine(LogCategory.General, "---Graph Test 2 Start---");
            GraphGenerator graphGen = new GraphGenerator(1280, 720, 15, 15);
            graph = graphGen.Graph;
            PathPlanner.Graph = graph;

            Log.WriteLine(LogCategory.General, string.Format("Graph Node Count {0}", graph.NodeCount.ToString()));
            foreach (GraphEdge edge in graph.Edges())
            {
                Log.WriteLine(LogCategory.General, string.Format(" Edge From:{0} To:{1}", edge.From, edge.To));
            }
            Log.WriteLine(LogCategory.General, "---Graph Test 2 End---");
        }

        /// <summary>
        /// Allows the game to perform any initialization it needs to before starting to run.
        /// This is where it can query for any required services and load any non-graphic
        /// related content.  Calling base.Initialize will enumerate through any components
        /// and initialize them as well.
        /// </summary>
        protected override void Initialize()
        {
            SetScreen(1280, 720, false);
            sceneManager = new SceneManager(this);
            uiManager = new UIManager(this);
            entityManager = new EntityManager(this);
            inputManager = new InputManager(this);
            fontManager = new FontManager();
            base.Initialize();
        }

        /// <summary>
        /// LoadContent will be called once per game and is the place to load
        /// all of your content.
        /// </summary>
        protected override void LoadContent()
        {
            spriteBatch = new SpriteBatch(GraphicsDevice);
            
            // Load Font
            defaultFont = Content.Load<SpriteFont>("Fonts\\DefaultFont");
            fontManager.AddFont(defaultFont, "Default");

            // Load Sprites
            CustomerSpriteContent.LoadContent(Content);

            // Add Services
            Services.AddService(typeof(SceneManager), sceneManager);
            Services.AddService(typeof(UIManager), uiManager);
            Services.AddService(typeof(EntityManager), entityManager);
            Services.AddService(typeof(FontManager), fontManager);
            Services.AddService(typeof(InputManager), inputManager);

            // Add Components
            Components.Add(sceneManager);
            Components.Add(uiManager);
            Components.Add(entityManager);
            Components.Add(inputManager);

            GraphGenerator graphGen = new GraphGenerator(1280, 720, 20, 20);
            graph = graphGen.Graph;
            PathPlanner.Graph = graph;
            

            //***UI***
            uiManager.AddUserInterface("MainMenu", new MainMenuUI(this));
            uiManager.AddUserInterface("Start", new PressStartUI(this));
            uiManager.AddUserInterface("Exit", new ExitGameUI(this));
            uiManager.AddUserInterface("Pause", new PauseUI(this));
            uiManager.AddUserInterface("Stage", new StageUI(this));
            uiManager.AddUserInterface("Graph", new GraphDisplayUI(this));
            uiManager.AddUserInterface("Help", new HelpUI(this));
      
            
            // Scenes
            sceneManager.AddScene("MainMenu", new MainMenuScene(this));
            sceneManager.AddScene("Play", new PlayScene(this));
            sceneManager.AddScene("Help", new HelpScene(this));
          


            sceneManager.ChangeScene("MainMenu");
            uiManager.PushUserInterface("Start");

            
            
       
        }

        /// <summary>
        /// UnloadContent will be called once per game and is the place to unload
        /// all content.
        /// </summary>
        protected override void UnloadContent()
        {
          
        }

        /// <summary>
        /// Allows the game to run logic such as updating the world,
        /// checking for collisions, gathering input, and playing audio.
        /// </summary>
        /// <param name="gameTime">Provides a snapshot of timing values.</param>
        protected override void Update(GameTime gameTime)
        {
            // Allows the game to exit
            if (GamePad.GetState(PlayerIndex.One).Buttons.Back == ButtonState.Pressed ||
                Keyboard.GetState().IsKeyDown(Keys.Escape))
                this.Exit();

            if (Keyboard.GetState().IsKeyDown(Keys.F))
            {
                graphics.ToggleFullScreen();
            }

            base.Update(gameTime);
        }

    
        protected override void Draw(GameTime gameTime)
        {
            GraphicsDevice.Clear(Color.White);
            base.Draw(gameTime);
        }

        // Helper Method to set resolution and fullscreen properties.
        public void SetScreen(int width, int height, bool fullScreen)
        {
            graphics.PreferredBackBufferWidth = width;
            graphics.PreferredBackBufferHeight = height;
            graphics.IsFullScreen = fullScreen;
            Rectangle screenRect = new Rectangle(0, 0, width, height);
            Parameters.SetParam("ScreenRect", screenRect);
            graphics.ApplyChanges();
        }

        

        #region Entry Point
        static void Main(string[] args)
        {
            using (CashierGame game = new CashierGame())
            {
                _instance = game;
                game.Run();
            }

#if !XBOX
            //Log.Save(@"log");
#endif
        }

        private static CashierGame _instance;
        public static CashierGame Instance
        {
            get { return _instance; }
        }

        #endregion
    }
}
